/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/


#include "mx_auto_config.h"
#include "myriexpress.h"
#if MX_OS_WINNT
#include "getopt.h"
#include <winsock2.h>
#define sleep(x_) Sleep((x_)*1000)
#else
#include <unistd.h>
#include <netinet/in.h>
#endif
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include <errno.h>

#include "mx__lib_types.h"
#include "mx__lib.h"
#include "mx__driver_interface.h"
#include "mx__fops.h"

/* make sure counters for 1-port board are index 0 and 1 */
#define SEND0_INDEX 0
#define RECV0_INDEX 1
#define SEND1_INDEX 2
#define RECV1_INDEX 3

#if MX_DRIVER_API_MAGIC < 0x500
typedef struct {
  uint32_t board_number;
  uint32_t count;
  uint64_t indexes;
  uint64_t values;
} mx_get_some_counters_t;

static int
mx__get_some_counters(mx_endpt_handle_t fd, mx_get_some_counters_t *c)
{
  uint32_t cnt[256];
  uint32_t *indexes = (void*)(uintptr_t)c->indexes;
  uint32_t *values = (void*)(uintptr_t)c->values;
  int i;
  mx_get_info(NULL, MX_COUNTERS_VALUES, &c->board_number, sizeof(c->board_number),
	      cnt, sizeof(cnt));
  for (i=0;i<c->count;i++)
    values[i] = htonl(cnt[indexes[i]]);
  return MX_SUCCESS;
}
#endif

void
usage()
{
  fprintf(stderr, "Usage: mx_bandwidth [args]\n");
  fprintf(stderr, "-b - Board number [0]\n");
  fprintf(stderr, "-t - Time interval (seconds) [1]\n");
  fprintf(stderr, "-k - Keep running\n");
  fprintf(stderr, "-p - Display counters for each port on multiple ports boards\n");
  fprintf(stderr, "-a - Also display aggregated counters for each iteration\n");
  fprintf(stderr, "-A - Only display aggregated counters for each iteration\n");
  fprintf(stderr, "-h - help\n");
}

int 
main(int argc, char **argv)
{
  mx_endpt_handle_t fd;
  mx_return_t ret;
  uint32_t counters[4];
  uint32_t indexes[4];
  uint32_t board_id, count;
  uint32_t delay;
  int keep, print_each_port, print_aggregate, print_immediate;
  uint32_t i; 
  uint32_t send0_start, send0_prev, send0_new;
  uint32_t send1_start, send1_prev, send1_new;
  uint32_t recv0_start, recv0_prev, recv0_new;
  uint32_t recv1_start, recv1_prev, recv1_new;
  uint32_t send0, send1, send;
  uint32_t recv0, recv1, recv;
  uint64_t send_accum = 0, recv_accum = 0, time_accum = 0;
  int found_send0, found_send1, found_recv0, found_recv1;
  int two_ports;
  extern char *optarg;
  int c;
  mx_get_counters_strings_t get_count, *get_strings;
  mx_get_some_counters_t get_some;

  board_id = 0;
  delay = 1;
  keep = 0;
  print_each_port = 0;
  print_aggregate = 0;
  print_immediate = 1;
  while ((c = getopt(argc, argv, "hb:t:kpaA"))!= EOF) switch(c) {
  case 'b':
    board_id = atoi(optarg);
    break;
  case 't':
    delay = atoi(optarg);
    break;
  case 'k':
    keep = 1;
    break;
  case 'p':
    print_each_port = 1;
    break;
  case 'a':
    print_aggregate = 1;
    break;
  case 'A':
    print_aggregate = 1;
    print_immediate = 0;
    break;
  case 'h':
  default:
    usage();
    exit(1);
  }
  
  mx_init();
  ret = mx_open_any_board(&fd);
  if (ret != MX_SUCCESS) {
    printf("open failed: %s\n", mx_strerror(ret));
    return 0;
  }

  get_count.count = 0;
  get_count.board_number = board_id;
  ret = mx__get_counters_strings(fd, board_id, &get_count);
  if (ret != MX_SUCCESS) {
    perror("get counters count failed");
  }
  count = get_count.count;

  get_strings = malloc (offsetof(mx_get_counters_strings_t, label)
			+ MX_MAX_STR_LEN * count);
  if (get_strings == NULL) {
    printf("out of memory\n");
    goto abort_with_fd;
  }

  get_strings->count = count;
  get_strings->board_number = board_id;
  ret = mx__get_counters_strings(fd, board_id, get_strings);
  if (ret != MX_SUCCESS) {
    perror("get counters strings failed");
    goto abort_with_fd;
  }

  found_send0 = -1;
  found_send1 = -1;
  found_recv0 = -1;
  found_recv1 = -1;
  for (i = 0; i < count; ++i) {
    char * name = ((char*)&get_strings->label) + i * MX_MAX_STR_LEN;
    if (!strcmp(name, "Net send KBytes (Port 0)") ||
	!strcmp(name, "Net send KBytes")) {
      found_send0 = i;
    } else if (!strcmp(name, "Net send KBytes (Port 1)")) {
      found_send1 = i;
    } else if (!strcmp(name, "Net recv KBytes (Port 0)") ||
	       !strcmp(name, "Net recv KBytes")) {
      found_recv0 = i;
    } else if (!strcmp(name, "Net recv KBytes (Port 1)")) {
      found_recv1 = i;
    }
  }
  if (found_send0 < 0 || found_recv0 < 0) {
    printf("failed to find byte counters\n");
    goto abort_with_fd;
  }
  if ((found_send1 < 0) ^ (found_recv1 < 0)) {
    printf("found only one byte counter for port 1\n");
    goto abort_with_fd;
  }
  two_ports = found_send1 >= 0;

  get_some.board_number = board_id;
  get_some.count = 2 + 2*two_ports;
  get_some.indexes = (uintptr_t) indexes;
  get_some.values = (uintptr_t) counters;

  indexes[SEND0_INDEX] = found_send0;
  indexes[RECV0_INDEX] = found_recv0;
  indexes[SEND1_INDEX] = found_send1;
  indexes[RECV1_INDEX] = found_recv1;

  do {
    ret = mx__get_some_counters(fd, &get_some);
    if ((ret != MX_SUCCESS) && (errno != EBUSY)) {
      perror("get some counters failed");
      goto abort_with_fd;
    }
  } while (ret);

  send0_prev = send0_start = htonl(counters[SEND0_INDEX]);
  recv0_prev = recv0_start = htonl(counters[RECV0_INDEX]);
  if (two_ports) {
    send1_prev = send1_start = htonl(counters[SEND1_INDEX]);
    recv1_prev = recv1_start = htonl(counters[RECV1_INDEX]);
  }

loop:
  sleep(delay);
  
  do {
    ret = mx__get_some_counters(fd, &get_some);
    if ((ret != MX_SUCCESS) && (errno != EBUSY)) {
      perror("get some counters failed");
      goto abort_with_fd;
    }
  } while (ret);

  send0_new = htonl(counters[SEND0_INDEX]);
  recv0_new = htonl(counters[RECV0_INDEX]);
  if (two_ports) {
    send1_new = htonl(counters[SEND1_INDEX]);
    recv1_new = htonl(counters[RECV1_INDEX]);
  }

  /* Compute and display counters for the last time interval */
  send0 = send0_new - send0_prev;
  recv0 = recv0_new - recv0_prev;
  if (found_recv1 >= 0) {
    send1 = send1_new - send1_prev;
    recv1 = recv1_new - recv1_prev;
    if (print_each_port && print_immediate) {
      printf("Port 0: sent %d kB (%.2f MB/s) \treceived %d kB (%.2f MB/s)\n",
	     (unsigned int) send0, ((float) send0)/1000/delay,
	     (unsigned int) recv0, ((float) recv0)/1000/delay);
      printf("Port 1: sent %d kB (%.2f MB/s) \treceived %d kB (%.2f MB/s)\n",
	     (unsigned int) send1, ((float) send1)/1000/delay,
	     (unsigned int) recv1, ((float) recv1)/1000/delay);
    }
  } else {
    send1 = recv1 = 0;
  }
  send = send0 + send1;
  recv = recv0 + recv1;
  if (print_immediate) {
    printf("Total : sent %d kB (%.2f MB/s) \treceived %d kB (%.2f MB/s)\n",
	   (unsigned int) send, ((float) send)/1000/delay,
	   (unsigned int) recv, ((float) recv)/1000/delay);
  }

  /* Display aggregated counters */
  if (print_aggregate) {
    send_accum += send;
    recv_accum += recv;
    time_accum += delay;
    printf("Aggregated total : sent %lld kB (%.2f MB/s) \treceived %lld kB (%.2f MB/s)\n",
	   (unsigned long long) send_accum, ((float) send_accum)/1000/time_accum,
	   (unsigned long long) recv_accum, ((float) recv_accum)/1000/time_accum);
  }

  if (keep) {
    send0_prev = send0_new;
    recv0_prev = recv0_new;
    if (two_ports) {
      send1_prev = send1_new;
      recv1_prev = recv1_new;
    }
    goto loop;
  }

 abort_with_fd:
  mx__close(fd);
  mx_finalize();

  return 0;
}
